/*
 * ESP32 Grid Frequency Monitor v2.0
 * Ported from STM32 project with enhanced features
 * 
 * Features:
 * - High-precision frequency measurement using timer interrupts
 * - GPS 1PPS calibration for improved accuracy
 * - Stepper motor control for physical display
 * - MQTT publishing to robertrileyruark.com
 * - Web server dashboard with real-time updates
 * - Configurable parameters via config.h
 * 
 * Author: Ported from STM32 implementation
 * Date: 2024
 * WiFi: starlink / redmondwa
 * Last Updated: October 11, 2025 - Added MQTT username/password configuration with NVM storage
 */

#include "config.h"
#include <WiFi.h>
#include <PubSubClient.h>
#include <WebServer.h>
#include <ArduinoJson.h>
#include <SPIFFS.h>
#include <EEPROM.h>

// Network objects
WiFiClient espClient;
PubSubClient mqttClient(espClient);
WebServer server(WEB_SERVER_PORT);

// Global variables
volatile uint32_t lastAcTime = 0;
volatile uint32_t lastPpsTime = 0;
volatile uint32_t acPeriod = 0;
volatile uint32_t ppsPeriod = 0;
volatile bool ppsDetected = false;
volatile bool newPpsPeriod = false;
volatile bool isInitialRun = true;
volatile int cycleCount = 0;
volatile int ppsCycleCount = 60;
volatile unsigned long lastPpsTime_ms = 0;

// Validity tracking
bool gridSignalValid = false;
bool ppsSignalValid = false;
unsigned long lastGridSignalTime = 0;
unsigned long lastPpsSignalTime = 0;

// Frequency measurement
float acFrequency = NOMINAL_FREQUENCY * 1000.0;  // mHz (60,000 for 60Hz)
float ppsFrequency = 80.0;     // MHz
float filteredAcFreq = NOMINAL_FREQUENCY * 1000.0; // mHz (60,000 for 60Hz)
//float filteredPpsFreq = 80.0;// MHz

// Stepper motor control
int currentPosition = 0;
int desiredPosition = 0;
int stepperState = 0;

// Sample buffer for averaging
uint32_t sampleBuffer[SAMPLE_COUNT];
int sampleIndex = 0;

// PPS sample buffer for averaging
uint32_t ppsSampleBuffer[SAMPLE_COUNT];
int ppsSampleIndex = 0;

// Timing variables
unsigned long lastPublishTime = 0;
bool publishOnNewPps = false;

// Device name storage
String deviceName = "Grid Frequency Monitor";
#define DEVICE_NAME_ADDR 0
#define DEVICE_NAME_MAX_LEN 32

// WiFi credentials storage
String wifiSSID = WIFI_SSID;
String wifiPassword = WIFI_PASSWORD;
#define WIFI_SSID_ADDR 64
#define WIFI_PASSWORD_ADDR 128
#define WIFI_CRED_MAX_LEN 32

// MQTT topic storage
String mqttTopic = MQTT_TOPIC;
#define MQTT_TOPIC_ADDR 192
#define MQTT_TOPIC_MAX_LEN 32

// MQTT credentials storage
String mqttUsername = "";
String mqttPassword = "";
#define MQTT_USERNAME_ADDR 224
#define MQTT_PASSWORD_ADDR 288
#define MQTT_CRED_MAX_LEN 32

// AP Mode configuration
bool apMode = false;
#define AP_SSID "GridFreq-Config"
#define AP_PASSWORD "config123"

// GPS NMEA data
String gpsTime = "";
String gpsDate = "";
bool gpsTimeValid = false;
String nmeaBuffer = "";

// MQTT Client ID with MAC address
String mqttClientId = "";

// Function prototypes
void setupWiFi();
void setupMQTT();
void setupWebServer();
void setupPins();
void setupInterrupts();
void loadDeviceName();
void saveDeviceName();
void loadWiFiCredentials();
void saveWiFiCredentials();
void loadMQTTTopic();
void saveMQTTTopic();
void loadMQTTCredentials();
void saveMQTTCredentials();
void setupAPMode();
void handleAPRoot();
void setupGPS();
void parseNMEA();
void processGPRMC(String sentence);
void testMQTTConnection();
void measureFrequency();
void controlStepper();
void stepUp();
void stepDown();
void zeroScale();
void publishFrequency();
void handleRoot();
void handleAPI();
void handleConfig();
void handleTest();
void handleSetDeviceName();
void handleSetWiFiCredentials();
void handleSetMQTTTopic();
void handleSetMQTTCredentials();
void reconnectMQTT();
void debugPrint(const String& message);

void setup() {
  Serial.begin(SERIAL_BAUD);
  debugPrint("ESP32 Grid Frequency Monitor v2.0 Starting...");
  
  // Initialize SPIFFS for web server
  if (!SPIFFS.begin(true)) {
    debugPrint("SPIFFS Mount Failed");
    return;
  }
  
  // Create unique MQTT Client ID using MAC address
  uint8_t mac[6];
  WiFi.macAddress(mac);
  mqttClientId = String(MQTT_CLIENT_ID) + "_" + 
                 String(mac[3], HEX) + String(mac[4], HEX) + String(mac[5], HEX);
  debugPrint("MQTT Client ID: " + mqttClientId);
  
  setupPins();
  setupInterrupts();
  setupGPS();
  loadDeviceName();
  loadWiFiCredentials();
  loadMQTTTopic();
  loadMQTTCredentials();
  setupWiFi();
  setupMQTT();
  testMQTTConnection();
  setupWebServer();
  
  // Initialize sample buffer
  for (int i = 0; i < SAMPLE_COUNT; i++) {
    sampleBuffer[i] = 16667; // Default 60Hz period in microseconds (1,000,000/60)
  }
  
  // Initialize PPS sample buffer
  for (int i = 0; i < SAMPLE_COUNT; i++) {
    ppsSampleBuffer[i] = 1000000; // Default 1Hz (1 second) period in microseconds
  }
  
  // Calibrate stepper motor
  debugPrint("Calibrating stepper motor...");
  zeroScale();
  
  debugPrint("Setup complete. Monitoring frequency...");
  debugPrint("Web interface: http://" + WiFi.localIP().toString());
}

void loop() {
  // Handle MQTT reconnection
  if (!mqttClient.connected()) {
    reconnectMQTT();
  }
  mqttClient.loop();
  
  // Handle web server
  server.handleClient();
  
  // Parse GPS NMEA data
  parseNMEA();
  
  // Measure and process frequency
  measureFrequency();
  
  // Control stepper motor
  controlStepper();
  
  // Publish frequency data periodically or when new PPS is detected
  if ((millis() - lastPublishTime >= PUBLISH_INTERVAL) || publishOnNewPps) {
    publishFrequency();
    lastPublishTime = millis();
    publishOnNewPps = false; // Clear flag after publishing
  }
  
  // Toggle status LED
  //digitalWrite(STATUS_LED, !digitalRead(STATUS_LED));
  
  delay(50);
}

void setupPins() {
  // Configure input pins with pull-up
  pinMode(AC_FREQ_PIN, INPUT_PULLUP);
  pinMode(GPS_PPS_PIN, INPUT_PULLUP);
  
  // Configure stepper motor pins
  pinMode(STEPPER_PIN1, OUTPUT);
  pinMode(STEPPER_PIN2, OUTPUT);
  pinMode(STEPPER_PIN3, OUTPUT);
  pinMode(STEPPER_PIN4, OUTPUT);
  
  // Configure status LED
  pinMode(STATUS_LED, OUTPUT);
  digitalWrite(STATUS_LED, HIGH);
  
  // Initialize stepper motor pins
  digitalWrite(STEPPER_PIN1, LOW);
  digitalWrite(STEPPER_PIN2, LOW);
  digitalWrite(STEPPER_PIN3, LOW);
  digitalWrite(STEPPER_PIN4, LOW);
  
  debugPrint("Pins configured");
}

void setupInterrupts() {
  // Attach interrupts for frequency measurement
  attachInterrupt(digitalPinToInterrupt(AC_FREQ_PIN), acFreqISR, RISING);
  attachInterrupt(digitalPinToInterrupt(GPS_PPS_PIN), ppsISR, RISING);
  debugPrint("Interrupts configured");
}

void acFreqISR() {
  uint32_t currentTime = micros();
  if (lastAcTime > 0) {
    acPeriod = currentTime - lastAcTime;
    cycleCount++;
    lastGridSignalTime = millis();
    gridSignalValid = true;
  }
  lastAcTime = currentTime;
}

void ppsISR() {
  uint32_t currentTime = micros();
  lastPpsTime_ms = millis();
  lastPpsSignalTime = millis();
  ppsSignalValid = true;
  
  if (lastPpsTime > 0) {
    ppsPeriod = currentTime - lastPpsTime;
    ppsDetected = true;
    newPpsPeriod = true;
    
    if (!isInitialRun) {
      cycleCount -= 60; // Adjust for 1-second measurement window
      if (cycleCount - CYCLE_RESET_THRESHOLD > ppsCycleCount) {
        cycleCount = 0; // Reset if PPS is lost
        debugPrint("PPS lost - resetting cycle count");
      } else {
        ppsCycleCount = cycleCount;
      }
    } else {
      isInitialRun = false;
      cycleCount = 0;
      debugPrint("Initial PPS detected");
    }
  }
  lastPpsTime = currentTime;
}

void measureFrequency() {
  // Check for signal timeouts
  if (millis() - lastPpsTime_ms > PPS_TIMEOUT_MS) {
    ppsDetected = false;
    ppsSignalValid = false;
  }
  
  if (millis() - lastGridSignalTime > 100) { // 0.1 second timeout for grid signal
    gridSignalValid = false;
  }
  
  // Calculate AC frequency from period
  if (acPeriod > 0 && acPeriod < MAX_PERIOD && acPeriod > MIN_PERIOD) {
    // Store sample in buffer
    sampleBuffer[sampleIndex] = acPeriod;
    sampleIndex = (sampleIndex + 1) % SAMPLE_COUNT;
    
    // Calculate average period
    uint32_t sum = 0;
    for (int i = 0; i < SAMPLE_COUNT; i++) {
      sum += sampleBuffer[i];
    }
    
    // Calculate frequency in mHz
    acFrequency = (float)F_CLK / (float)(sum / SAMPLE_COUNT) * 1000.0 / ppsFrequency;
    
    // Apply low-pass filter
    filteredAcFreq = ALPHA * acFrequency + (1.0 - ALPHA) * filteredAcFreq;
  }
  
  // Update PPS frequency only when a new period is detected
  if (newPpsPeriod && ppsPeriod > 0 && ppsPeriod < MAX_PERIOD_PPS) {
    // Store PPS sample in buffer
    ppsSampleBuffer[ppsSampleIndex] = ppsPeriod;
    ppsSampleIndex = (ppsSampleIndex + 1) % SAMPLE_COUNT;
    
    // Calculate average PPS period
    uint32_t ppsSum = 0;
    for (int i = 0; i < SAMPLE_COUNT; i++) {
      ppsSum += ppsSampleBuffer[i];
    }
    
    // Calculate PPS frequency in MHz from average period
    ppsFrequency = (float)F_CLK / (float)(ppsSum / SAMPLE_COUNT);
    
    // Trigger MQTT publish on new PPS detection
    publishOnNewPps = true;
    
    newPpsPeriod = false; // Clear flag after processing
  }
}

void controlStepper() {
  // Calculate desired position (similar to STM32 code)
  // For 60Hz: 60,000 mHz is center position
  desiredPosition = (int)((filteredAcFreq - (NOMINAL_FREQUENCY * 1000.0)) * POSITION_SCALE);
  
  // Limit position range
  desiredPosition = constrain(desiredPosition, -1000, 1000);
  
  // Move stepper motor
  if (desiredPosition > currentPosition) {
    stepUp();
  } else if (desiredPosition < currentPosition) {
    stepDown();
  }
}

void stepUp() {
  switch (stepperState) {
    case 0:
      digitalWrite(STEPPER_PIN2, HIGH);
      digitalWrite(STEPPER_PIN4, LOW);
      delay(STEP_DELAY);
      stepperState = 3;
      break;
    case 1:
      digitalWrite(STEPPER_PIN3, HIGH);
      digitalWrite(STEPPER_PIN1, LOW);
      delay(STEP_DELAY);
      stepperState = 0;
      break;
    case 2:
      digitalWrite(STEPPER_PIN4, HIGH);
      digitalWrite(STEPPER_PIN2, LOW);
      delay(STEP_DELAY);
      stepperState = 1;
      break;
    case 3:
      digitalWrite(STEPPER_PIN1, HIGH);
      digitalWrite(STEPPER_PIN3, LOW);
      delay(STEP_DELAY);
      stepperState = 2;
      break;
  }
  currentPosition++;
}

void stepDown() {
  switch (stepperState) {
    case 0:
      digitalWrite(STEPPER_PIN1, HIGH);
      digitalWrite(STEPPER_PIN3, LOW);
      delay(STEP_DELAY);
      stepperState = 1;
      break;
    case 1:
      digitalWrite(STEPPER_PIN2, HIGH);
      digitalWrite(STEPPER_PIN4, LOW);
      delay(STEP_DELAY);
      stepperState = 2;
      break;
    case 2:
      digitalWrite(STEPPER_PIN3, HIGH);
      digitalWrite(STEPPER_PIN1, LOW);
      delay(STEP_DELAY);
      stepperState = 3;
      break;
    case 3:
      digitalWrite(STEPPER_PIN4, HIGH);
      digitalWrite(STEPPER_PIN2, LOW);
      delay(STEP_DELAY);
      stepperState = 0;
      break;
  }
  currentPosition--;
}

void zeroScale() {
  debugPrint("Zeroing stepper motor...");
  // Move stepper to zero position
  for (int i = 0; i < ZERO_STEPS_DOWN; i++) {
    stepDown();
  }
  for (int i = 0; i < ZERO_STEPS_UP; i++) {
    stepUp();
  }
  currentPosition = 0;
  debugPrint("Stepper motor zeroed");
}

void setupWiFi() {
  WiFi.begin(wifiSSID.c_str(), wifiPassword.c_str());
  debugPrint("Connecting to WiFi: " + wifiSSID);
  
  int attempts = 0;
  while (WiFi.status() != WL_CONNECTED && attempts < 20) {
    delay(500);
    debugPrint(".");
    attempts++;
  }
  
  if (WiFi.status() == WL_CONNECTED) {
    debugPrint("WiFi connected");
    debugPrint("IP address: " + WiFi.localIP().toString());
    apMode = false;
  } else {
    debugPrint("WiFi connection failed - Starting AP mode");
    setupAPMode();
  }
}

void setupMQTT() {
  mqttClient.setServer(MQTT_SERVER, MQTT_PORT);
  debugPrint("MQTT configured");
}

void setupWebServer() {
  if (apMode) {
    // AP Mode routes
    server.on("/", handleAPRoot);
    server.on("/setwifi", HTTP_POST, handleSetWiFiCredentials);
    server.onNotFound([]() {
      server.sendHeader("Location", "/", true);
      server.send(302, "text/plain", "");
    });
  } else {
    // Normal mode routes
    server.on("/", handleRoot);
    server.on("/api", handleAPI);
    server.on("/config", handleConfig);
    server.on("/test", handleTest);
    server.on("/setdevicename", HTTP_POST, handleSetDeviceName);
    server.on("/setwifi", HTTP_POST, handleSetWiFiCredentials);
    server.on("/setmqtttopic", HTTP_POST, handleSetMQTTTopic);
    server.on("/setmqttcreds", HTTP_POST, handleSetMQTTCredentials);
  }
  server.begin();
  debugPrint("Web server started on port " + String(WEB_SERVER_PORT));
}

// WebSocket functions removed - using polling instead

void handleRoot() {
  String html = "<!DOCTYPE html><html><head><title>Grid Frequency Monitor v2.0</title>";
  html += "<meta name='viewport' content='width=device-width, initial-scale=1.0'>";
  html += "<style>";
  html += "body{font-family:Arial,sans-serif;text-align:center;margin:10px;background:#f0f0f0;}";
  html += ".container{max-width:800px;margin:0 auto;background:white;padding:20px;border-radius:10px;box-shadow:0 2px 10px rgba(0,0,0,0.1);}";
  html += ".frequency{font-size:48px;color:#0066cc;margin:20px;font-weight:bold;}";
  html += ".gauge{width:300px;height:150px;border:3px solid #333;border-radius:150px 150px 0 0;margin:20px auto;position:relative;background:linear-gradient(90deg,#ff0000 0%,#ffff00 35%,#00ff00 50%,#ffff00 65%,#ff0000 100%);overflow:hidden;}";
  html += ".needle{width:3px;height:120px;background:#000;position:absolute;bottom:0;left:50%;transform-origin:bottom;transform:translateX(-50%) rotate(-90deg);border-radius:2px;z-index:10;}";
  html += ".info{display:flex;flex-wrap:wrap;justify-content:space-around;margin:20px 0;}";
  html += ".info-box{background:#f8f8f8;padding:15px;border-radius:5px;border-left:4px solid #0066cc;margin:5px;min-width:150px;flex:1;}";
  html += ".error{color:#ff0000;font-weight:bold;}";
  html += ".success{color:#00aa00;font-weight:bold;}";
  html += ".loading{color:#666;font-style:italic;}";
  html += ".status{position:fixed;top:10px;right:10px;padding:5px 10px;border-radius:5px;font-size:12px;background:#00aa00;color:white;}";
  html += "@media (max-width: 600px) {.info{flex-direction:column;}.gauge{width:250px;height:125px;}}";
  html += "</style></head><body>";
  html += "<div class='status'>Live Updates</div>";
  html += "<div class='container'>";
  html += "<h1>" + deviceName + " Grid Frequency</h1>";
  html += "<div class='gauge'><div class='needle' id='needle'></div></div>";
  html += "<div class='frequency' id='freq'>-- Hz</div>";
  html += "<div class='info'>";
  html += "<div class='info-box'><strong>GPS PPS:</strong><br><span id='pps' class='loading'>Loading...</span></div>";
  html += "<div class='info-box'><strong>GPS Time:</strong><br><span id='gps-time' class='loading'>--</span></div>";
  html += "<div class='info-box'><strong>GPS Date:</strong><br><span id='gps-date' class='loading'>--</span></div>";
  html += "<div class='info-box'><strong>Position:</strong><br><span id='pos'>0</span></div>";
  html += "<div class='info-box'><strong>Raw Frequency:</strong><br><span id='raw' class='loading'>-- Hz</span></div>";
  html += "<div class='info-box'><strong>Internal Oscillator:</strong><br><span id='pps_freq' class='loading'>-- MHz</span></div>";
  html += "</div>";
  html += "<p><a href='/config'>Configuration</a> | <a href='/api'>API Data</a> | <a href='/test'>Test</a></p>";
  html += "</div>";
  html += "<script>";
  html += "function updateData() {";
  html += "  fetch('/api').then(response => {";
  html += "    if (!response.ok) throw new Error('Network response was not ok');";
  html += "    return response.json();";
  html += "  }).then(data => {";
  html += "    console.log('Received data:', data);";
  html += "    if (data.frequency !== undefined && data.frequency > 0) {";
  html += "      if (data.pps_detected) {";
  html += "        document.getElementById('freq').textContent = data.frequency.toFixed(3) + ' Hz';";
  html += "      } else {";
  html += "        document.getElementById('freq').textContent = data.frequency.toFixed(3) + ' Hz (Raw)';";
  html += "      }";
  html += "    } else {";
  html += "      document.getElementById('freq').textContent = '-- Hz';";
  html += "    }";
  html += "    if (data.raw_frequency !== undefined && data.raw_frequency > 0) {";
  html += "      document.getElementById('raw').textContent = data.raw_frequency.toFixed(3) + ' Hz';";
  html += "    } else {";
  html += "      document.getElementById('raw').textContent = '-- Hz';";
  html += "    }";
  html += "    if (data.pps_frequency !== undefined && data.pps_frequency > 0) {";
  html += "      document.getElementById('pps_freq').textContent = data.pps_frequency.toFixed(6) + ' MHz';";
  html += "    } else {";
  html += "      document.getElementById('pps_freq').textContent = '-- MHz';";
  html += "    }";
  html += "    const ppsElement = document.getElementById('pps');";
  html += "    if (data.pps_detected) {";
  html += "      ppsElement.innerHTML = 'Detected';";
  html += "      ppsElement.className = 'success';";
  html += "    } else {";
  html += "      ppsElement.innerHTML = 'Not detected';";
  html += "      ppsElement.className = 'error';";
  html += "    }";
    html += "    if (data.position !== undefined) {";
    html += "      document.getElementById('pos').textContent = data.position;";
    html += "    }";
    html += "    if (data.gps_time !== undefined && data.gps_time !== '') {";
    html += "      document.getElementById('gps-time').textContent = data.gps_time;";
    html += "      document.getElementById('gps-time').className = 'success';";
    html += "    } else {";
    html += "      document.getElementById('gps-time').textContent = '--';";
    html += "      document.getElementById('gps-time').className = 'error';";
    html += "    }";
    html += "    if (data.gps_date !== undefined && data.gps_date !== '') {";
    html += "      document.getElementById('gps-date').textContent = data.gps_date;";
    html += "      document.getElementById('gps-date').className = 'success';";
    html += "    } else {";
    html += "      document.getElementById('gps-date').textContent = '--';";
    html += "      document.getElementById('gps-date').className = 'error';";
    html += "    }";
  html += "    if (data.frequency !== undefined && data.frequency > 0) {";
  html += "      const angle = (data.frequency - 60) * 180;";
  html += "      document.getElementById('needle').style.transform = 'translateX(-50%) rotate(' + angle + 'deg)';";
  html += "      document.getElementById('needle').style.display = 'block';";
  html += "    } else {";
  html += "      document.getElementById('needle').style.display = 'none';";
  html += "    }";
  html += "  }).catch(error => {";
  html += "    console.error('Error fetching data:', error);";
  html += "    document.getElementById('freq').textContent = 'Error';";
  html += "  });";
  html += "}";
  html += "setInterval(updateData, 500);";
  html += "updateData();";
  html += "</script></body></html>";

  server.send(200, "text/html", html);
}


void handleAPI() {
  StaticJsonDocument<400> doc;
  
  // Use the working frequency values directly
  float displayFreq = filteredAcFreq / 1000.0; // Convert mHz to Hz
  float rawFreq = acFrequency / 1000.0;
  float ppsFreq = ppsFrequency;
  bool showNeedle = gridSignalValid;
  
  // Debug output
  debugPrint("API: displayFreq=" + String(displayFreq) + ", rawFreq=" + String(rawFreq) + ", ppsFreq=" + String(ppsFreq));
  debugPrint("API: ppsDetected=" + String(ppsDetected) + ", gridValid=" + String(gridSignalValid));
  
  doc["frequency"] = displayFreq;
  doc["raw_frequency"] = rawFreq;
  doc["pps_detected"] = ppsDetected && ppsSignalValid;
  doc["pps_frequency"] = ppsFreq;
  doc["grid_valid"] = gridSignalValid;
  doc["pps_valid"] = ppsSignalValid;
  doc["show_needle"] = showNeedle;
  doc["position"] = currentPosition;
  doc["cycle_count"] = cycleCount;
  doc["gps_time"] = gpsTime;
  doc["gps_date"] = gpsDate;
  doc["gps_time_valid"] = gpsTimeValid;
  doc["timestamp"] = millis();
  doc["uptime"] = millis() / 1000;
  doc["status"] = "ok";
  
  String response;
  serializeJson(doc, response);
  
  // Add CORS headers for better compatibility
  server.sendHeader("Access-Control-Allow-Origin", "*");
  server.sendHeader("Access-Control-Allow-Methods", "GET");
  server.send(200, "application/json", response);
}

void handleConfig() {
  String html = "<!DOCTYPE html><html><head><title>Configuration</title>";
  html += "<meta name='viewport' content='width=device-width, initial-scale=1.0'>";
  html += "<style>";
  html += "body{font-family:Arial,sans-serif;margin:20px;background:#f0f0f0;}";
  html += ".container{max-width:600px;margin:0 auto;background:white;padding:20px;border-radius:10px;box-shadow:0 2px 10px rgba(0,0,0,0.1);}";
  html += "h1{color:#0066cc;text-align:center;}";
  html += ".form-group{margin:15px 0;}";
  html += "label{display:block;margin-bottom:5px;font-weight:bold;}";
  html += "input[type='text'],input[type='number']{width:100%;padding:8px;border:1px solid #ddd;border-radius:4px;box-sizing:border-box;}";
  html += "button{background:#0066cc;color:white;padding:10px 20px;border:none;border-radius:4px;cursor:pointer;margin:5px;}";
  html += "button:hover{background:#0052a3;}";
  html += ".back{text-align:center;margin-top:20px;}";
  html += ".section{margin:20px 0;padding:15px;border:1px solid #ddd;border-radius:5px;}";
  html += ".section h3{margin-top:0;color:#0066cc;}";
  html += "</style></head><body>";
  html += "<div class='container'>";
  html += "<h1>Configuration</h1>";
  
  // Device Name Section
  html += "<div class='section'>";
  html += "<h3>Device Settings</h3>";
  html += "<form action='/setdevicename' method='POST'>";
  html += "<div class='form-group'>";
  html += "<label for='device_name'>Device Name:</label>";
  html += "<input type='text' id='device_name' name='device_name' value='" + deviceName + "' maxlength='" + String(DEVICE_NAME_MAX_LEN) + "'>";
  html += "</div>";
  html += "<button type='submit'>Save Device Name</button>";
  html += "</form>";
  html += "</div>";
  
  // WiFi Credentials Section
  html += "<div class='section'>";
  html += "<h3>WiFi Settings</h3>";
  html += "<form action='/setwifi' method='POST'>";
  html += "<div class='form-group'>";
  html += "<label for='wifi_ssid'>WiFi SSID:</label>";
  html += "<input type='text' id='wifi_ssid' name='wifi_ssid' value='" + wifiSSID + "' maxlength='" + String(WIFI_CRED_MAX_LEN) + "'>";
  html += "</div>";
  html += "<div class='form-group'>";
  html += "<label for='wifi_password'>WiFi Password:</label>";
  html += "<input type='password' id='wifi_password' name='wifi_password' value='" + wifiPassword + "' maxlength='" + String(WIFI_CRED_MAX_LEN) + "'>";
  html += "</div>";
  html += "<button type='submit'>Save WiFi Settings</button>";
  html += "</form>";
  html += "<p><em>Note: Device will restart after saving WiFi settings</em></p>";
  html += "</div>";
  
  // MQTT Topic Section
  html += "<div class='section'>";
  html += "<h3>MQTT Settings</h3>";
  html += "<form action='/setmqtttopic' method='POST'>";
  html += "<div class='form-group'>";
  html += "<label for='mqtt_topic'>MQTT Topic:</label>";
  html += "<input type='text' id='mqtt_topic' name='mqtt_topic' value='" + mqttTopic + "' maxlength='" + String(MQTT_TOPIC_MAX_LEN) + "'>";
  html += "</div>";
  html += "<button type='submit'>Save MQTT Topic</button>";
  html += "</form>";
  html += "<form action='/setmqttcreds' method='POST' style='margin-top:15px;'>";
  html += "<div class='form-group'>";
  html += "<label for='mqtt_username'>MQTT Username:</label>";
  html += "<input type='text' id='mqtt_username' name='mqtt_username' value='" + mqttUsername + "' maxlength='" + String(MQTT_CRED_MAX_LEN) + "'>";
  html += "</div>";
  html += "<div class='form-group'>";
  html += "<label for='mqtt_password'>MQTT Password:</label>";
  html += "<input type='password' id='mqtt_password' name='mqtt_password' value='" + mqttPassword + "' maxlength='" + String(MQTT_CRED_MAX_LEN) + "'>";
  html += "</div>";
  html += "<button type='submit'>Save MQTT Credentials</button>";
  html += "</form>";
  html += "<p><em>Note: Leave username empty for anonymous MQTT connection</em></p>";
  html += "</div>";
  
  // Current Settings Section
  html += "<div class='section'>";
  html += "<h3>Current Settings (Read-Only)</h3>";
  html += "<p><strong>WiFi SSID:</strong> " + String(WIFI_SSID) + "</p>";
  html += "<p><strong>MQTT Server:</strong> " + String(MQTT_SERVER) + "</p>";
  html += "<p><strong>MQTT Topic:</strong> " + mqttTopic + "</p>";
  html += "<p><strong>Sample Count:</strong> " + String(SAMPLE_COUNT) + "</p>";
  html += "<p><strong>Alpha (AC):</strong> " + String(ALPHA, 3) + "</p>";
  html += "<p><strong>Alpha (PPS):</strong> " + String(ALPHA_PPS, 3) + "</p>";
  html += "</div>";
  
  html += "<div class='back'>";
  html += "<a href='/'>Back to Dashboard</a>";
  html += "</div>";
  html += "</div></body></html>";
  
  server.send(200, "text/html", html);
}

void handleTest() {
  String html = "<!DOCTYPE html><html><head><title>System Test</title></head><body>";
  html += "<h1>System Test Page</h1>";
  html += "<h2>System Status</h2>";
  html += "<p><strong>WiFi Status:</strong> ";
  html += (WiFi.status() == WL_CONNECTED ? "Connected" : "Disconnected");
  html += "</p>";
  html += "<p><strong>IP Address:</strong> ";
  html += WiFi.localIP().toString();
  html += "</p>";
  html += "<p><strong>MQTT Status:</strong> ";
  html += (mqttClient.connected() ? "Connected" : "Disconnected");
  html += "</p>";
  html += "<p><strong>Uptime:</strong> ";
  html += String(millis() / 1000);
  html += " seconds</p>";
  html += "<p><strong>Free Heap:</strong> ";
  html += String(ESP.getFreeHeap());
  html += " bytes</p>";
  
  html += "<h2>Frequency Data</h2>";
  html += "<p><strong>Grid Signal Valid:</strong> ";
  html += (gridSignalValid ? "Yes" : "No");
  html += "</p>";
  html += "<p><strong>PPS Signal Valid:</strong> ";
  html += (ppsSignalValid ? "Yes" : "No");
  html += "</p>";
  html += "<p><strong>Filtered Frequency:</strong> ";
  html += String(filteredAcFreq / 1000.0, 3);
  html += " Hz</p>";
  html += "<p><strong>Raw Frequency:</strong> ";
  html += String(acFrequency / 1000.0, 3);
  html += " Hz</p>";
  html += "<p><strong>PPS Detected:</strong> ";
  html += (ppsDetected ? "Yes" : "No");
  html += "</p>";
  html += "<p><strong>Internal Oscillator:</strong> ";
  html += String(ppsFrequency, 6);
  html += " MHz</p>";
  html += "<p><strong>Stepper Position:</strong> ";
  html += String(currentPosition);
  html += "</p>";
  html += "<p><strong>Cycle Count:</strong> ";
  html += String(cycleCount);
  html += "</p>";
  html += "<p><strong>Last Grid Signal:</strong> ";
  html += String((millis() - lastGridSignalTime) / 1000);
  html += "s ago</p>";
  html += "<p><strong>Last PPS Signal:</strong> ";
  html += String((millis() - lastPpsSignalTime) / 1000);
  html += "s ago</p>";
  
  html += "<h2>GPIO Status</h2>";
  html += "<p><strong>AC Input (GPIO 2):</strong> ";
  html += String(digitalRead(AC_FREQ_PIN));
  html += "</p>";
  html += "<p><strong>PPS Input (GPIO 4):</strong> ";
  html += String(digitalRead(GPS_PPS_PIN));
  html += "</p>";
  html += "<p><strong>Status LED (GPIO 21):</strong> ";
  html += String(digitalRead(STATUS_LED));
  html += "</p>";
  
  html += "<h2>Test API</h2>";
  html += "<button onclick='testAPI()'>Test API Call</button>";
  html += "<div id='apiResult'></div>";
  
  html += "<script>";
  html += "function testAPI() {";
  html += "fetch('/api').then(r => r.json()).then(d => {";
  html += "document.getElementById('apiResult').innerHTML = '<pre>' + JSON.stringify(d, null, 2) + '</pre>';";
  html += "}).catch(e => {";
  html += "document.getElementById('apiResult').innerHTML = 'Error: ' + e.message;";
  html += "});";
  html += "}";
  html += "</script>";
  
  html += "<p><a href='/'>Back to Dashboard</a></p>";
  html += "</body></html>";
  
  server.send(200, "text/html", html);
}

void publishFrequency() {
  if (mqttClient.connected()) {
    //float deviation = filteredAcFreq - (NOMINAL_FREQUENCY * 1000.0); // Deviation in mHz (60,000 for 60Hz)
    
    StaticJsonDocument<500> doc;
    doc["device_name"] = deviceName;
    doc["frequency_hz"] = filteredAcFreq / 1000.0;
    //doc["deviation_mhz"] = deviation;
    doc["pps_detected"] = ppsDetected;
    doc["osc_error_ppm"] = (ppsFrequency / 80.0 - 1.0 )* 1000000.0;

    doc["gps_time"] = gpsTime;
    doc["gps_date"] = gpsDate;
    doc["gps_time_valid"] = gpsTimeValid;

    
    String payload;
    serializeJson(doc, payload);
    
    if (mqttClient.publish(mqttTopic.c_str(), payload.c_str())) {
      debugPrint("Published: " + payload);
    } else {
      debugPrint("MQTT publish failed - State: " + String(mqttClient.state()) + 
                ", Connected: " + String(mqttClient.connected()) + 
                ", Payload length: " + String(payload.length()) +
                ", Topic: " + mqttTopic);
      
      // Try to reconnect if publish fails
      if (!mqttClient.connected()) {
        debugPrint("MQTT disconnected, attempting reconnect...");
        reconnectMQTT();
      }
    }
  }
}

void reconnectMQTT() {
  int attempts = 0;
  while (!mqttClient.connected() && attempts < MAX_MQTT_RETRY) {
    debugPrint("Attempting MQTT connection to " + String(MQTT_SERVER) + ":" + String(MQTT_PORT));
    
    bool connected = false;
    if (mqttUsername.length() > 0) {
      // Use authentication if username is configured
      debugPrint("Connecting with username: " + mqttUsername + ", Client ID: " + mqttClientId);
      connected = mqttClient.connect(mqttClientId.c_str(), mqttUsername.c_str(), mqttPassword.c_str());
    } else {
      // Connect without authentication
      debugPrint("Connecting with Client ID: " + mqttClientId);
      connected = mqttClient.connect(mqttClientId.c_str());
    }
    
    if (connected) {
      debugPrint("MQTT connected successfully");
    } else {
      debugPrint("MQTT failed - State: " + String(mqttClient.state()) + 
                ", WiFi: " + String(WiFi.status()) + 
                ", Attempt: " + String(attempts + 1));
      delay(5000);
      attempts++;
    }
  }
  
  if (!mqttClient.connected()) {
    debugPrint("MQTT connection failed after " + String(MAX_MQTT_RETRY) + " attempts");
  }
}

// WebSocket functions removed - using fast polling instead

void loadDeviceName() {
  EEPROM.begin(512);
  String name = "";
  for (int i = 0; i < DEVICE_NAME_MAX_LEN; i++) {
    char c = EEPROM.read(DEVICE_NAME_ADDR + i);
    if (c == 0) break;
    name += c;
  }
  if (name.length() > 0) {
    deviceName = name;
  }
  debugPrint("Device name loaded: " + deviceName);
}

void saveDeviceName() {
  EEPROM.begin(512);
  for (int i = 0; i < DEVICE_NAME_MAX_LEN; i++) {
    if (i < deviceName.length()) {
      EEPROM.write(DEVICE_NAME_ADDR + i, deviceName.charAt(i));
    } else {
      EEPROM.write(DEVICE_NAME_ADDR + i, 0);
    }
  }
  EEPROM.commit();
  debugPrint("Device name saved: " + deviceName);
}

void loadWiFiCredentials() {
  EEPROM.begin(512);
  String ssid = "";
  String password = "";
  
  // Load SSID
  for (int i = 0; i < WIFI_CRED_MAX_LEN; i++) {
    char c = EEPROM.read(WIFI_SSID_ADDR + i);
    if (c == 0) break;
    ssid += c;
  }
  
  // Load Password
  for (int i = 0; i < WIFI_CRED_MAX_LEN; i++) {
    char c = EEPROM.read(WIFI_PASSWORD_ADDR + i);
    if (c == 0) break;
    password += c;
  }
  
  if (ssid.length() > 0) {
    wifiSSID = ssid;
  }
  if (password.length() > 0) {
    wifiPassword = password;
  }
  debugPrint("WiFi credentials loaded: " + wifiSSID);
}

void saveWiFiCredentials() {
  EEPROM.begin(512);
  
  // Save SSID
  for (int i = 0; i < WIFI_CRED_MAX_LEN; i++) {
    if (i < wifiSSID.length()) {
      EEPROM.write(WIFI_SSID_ADDR + i, wifiSSID.charAt(i));
    } else {
      EEPROM.write(WIFI_SSID_ADDR + i, 0);
    }
  }
  
  // Save Password
  for (int i = 0; i < WIFI_CRED_MAX_LEN; i++) {
    if (i < wifiPassword.length()) {
      EEPROM.write(WIFI_PASSWORD_ADDR + i, wifiPassword.charAt(i));
    } else {
      EEPROM.write(WIFI_PASSWORD_ADDR + i, 0);
    }
  }
  
  EEPROM.commit();
  debugPrint("WiFi credentials saved: " + wifiSSID);
}

void handleSetDeviceName() {
  if (server.hasArg("device_name")) {
    String newName = server.arg("device_name");
    if (newName.length() > 0 && newName.length() <= DEVICE_NAME_MAX_LEN) {
      deviceName = newName;
      saveDeviceName();
      debugPrint("Device name updated to: " + deviceName);
    }
  }
  server.sendHeader("Location", "/config");
  server.send(302, "text/plain", "");
}

void handleSetWiFiCredentials() {
  if (server.hasArg("wifi_ssid") && server.hasArg("wifi_password")) {
    String newSSID = server.arg("wifi_ssid");
    String newPassword = server.arg("wifi_password");
    
    if (newSSID.length() > 0 && newSSID.length() <= WIFI_CRED_MAX_LEN && 
        newPassword.length() <= WIFI_CRED_MAX_LEN) {
      wifiSSID = newSSID;
      wifiPassword = newPassword;
      saveWiFiCredentials();
      debugPrint("WiFi credentials updated: " + wifiSSID);
      
      // Send response before restarting
      String html = "<!DOCTYPE html><html><head><title>WiFi Updated</title>";
      html += "<meta http-equiv='refresh' content='10;url=/'>";
      html += "<style>body{font-family:Arial,sans-serif;text-align:center;margin:50px;background:#f0f0f0;}";
      html += ".container{max-width:400px;margin:0 auto;background:white;padding:30px;border-radius:10px;box-shadow:0 2px 10px rgba(0,0,0,0.1);}";
      html += "h1{color:#0066cc;} .success{color:#00aa00;font-size:18px;margin:20px 0;}";
      html += "</style></head><body>";
      html += "<div class='container'>";
      html += "<h1>WiFi Settings Updated</h1>";
      html += "<p class='success'>Device is restarting...</p>";
      html += "<p>Connecting to: " + wifiSSID + "</p>";
      if (apMode) {
        html += "<p>Switching from AP mode to WiFi mode...</p>";
      }
      html += "<p>Redirecting to dashboard in 10 seconds...</p>";
      html += "</div></body></html>";
      
      server.send(200, "text/html", html);
      
      // Restart after a short delay
      delay(2000);
      ESP.restart();
      return;
    }
  }
  
  // Redirect based on current mode
  if (apMode) {
    server.sendHeader("Location", "/");
  } else {
    server.sendHeader("Location", "/config");
  }
  server.send(302, "text/plain", "");
}

void setupAPMode() {
  apMode = true;
  WiFi.mode(WIFI_AP);
  WiFi.softAP(AP_SSID, AP_PASSWORD);
  debugPrint("AP Mode started");
  debugPrint("SSID: " + String(AP_SSID));
  debugPrint("Password: " + String(AP_PASSWORD));
  debugPrint("IP address: " + WiFi.softAPIP().toString());
}

void handleAPRoot() {
  String html = "<!DOCTYPE html><html><head><title>WiFi Configuration</title>";
  html += "<meta name='viewport' content='width=device-width, initial-scale=1.0'>";
  html += "<style>";
  html += "body{font-family:Arial,sans-serif;margin:20px;background:#f0f0f0;}";
  html += ".container{max-width:500px;margin:0 auto;background:white;padding:30px;border-radius:10px;box-shadow:0 2px 10px rgba(0,0,0,0.1);}";
  html += "h1{color:#0066cc;text-align:center;margin-bottom:30px;}";
  html += ".form-group{margin:20px 0;}";
  html += "label{display:block;margin-bottom:8px;font-weight:bold;color:#333;}";
  html += "input[type='text'],input[type='password']{width:100%;padding:12px;border:2px solid #ddd;border-radius:6px;box-sizing:border-box;font-size:16px;}";
  html += "input:focus{border-color:#0066cc;outline:none;}";
  html += "button{width:100%;background:#0066cc;color:white;padding:15px;border:none;border-radius:6px;cursor:pointer;font-size:16px;font-weight:bold;margin-top:10px;}";
  html += "button:hover{background:#0052a3;}";
  html += ".info{background:#e7f3ff;padding:15px;border-radius:6px;margin-bottom:20px;border-left:4px solid #0066cc;}";
  html += ".status{text-align:center;color:#666;margin-top:20px;}";
  html += "</style></head><body>";
  html += "<div class='container'>";
  html += "<h1>WiFi Configuration</h1>";
  html += "<div class='info'>";
  html += "<strong>Device:</strong> " + deviceName + "<br>";
  html += "<strong>Current Mode:</strong> Access Point<br>";
  html += "<strong>AP SSID:</strong> " + String(AP_SSID) + "<br>";
  html += "<strong>AP Password:</strong> " + String(AP_PASSWORD);
  html += "</div>";
  html += "<form action='/setwifi' method='POST'>";
  html += "<div class='form-group'>";
  html += "<label for='wifi_ssid'>WiFi Network Name (SSID):</label>";
  html += "<input type='text' id='wifi_ssid' name='wifi_ssid' placeholder='Enter WiFi network name' required>";
  html += "</div>";
  html += "<div class='form-group'>";
  html += "<label for='wifi_password'>WiFi Password:</label>";
  html += "<input type='password' id='wifi_password' name='wifi_password' placeholder='Enter WiFi password'>";
  html += "</div>";
  html += "<button type='submit'>Connect to WiFi</button>";
  html += "</form>";
  html += "<div class='status'>";
  html += "<p>After connecting, the device will restart and join your WiFi network.</p>";
  html += "</div>";
  html += "</div></body></html>";
  
  server.send(200, "text/html", html);
}

void setupGPS() {
  Serial2.begin(9600, SERIAL_8N1, GPS_UART_PIN, -1); // RX on GPS_UART_PIN, no TX
  debugPrint("GPS UART configured on GPIO " + String(GPS_UART_PIN));
}

void parseNMEA() {
  // Read available characters from GPS UART
  while (Serial2.available()) {
    char c = Serial2.read();
    
    if (c == '\n' || c == '\r') {
      if (nmeaBuffer.length() > 0) {
        // Process complete NMEA sentence
        if (nmeaBuffer.startsWith("$GPRMC") || nmeaBuffer.startsWith("$GNRMC")) {
          processGPRMC(nmeaBuffer);
        }
        nmeaBuffer = "";
      }
    } else if (c >= 32 && c <= 126) { // Printable ASCII characters
      nmeaBuffer += c;
      
      // Prevent buffer overflow
      if (nmeaBuffer.length() > 200) {
        nmeaBuffer = "";
      }
    }
  }
}

void processGPRMC(String sentence) {
  // Parse GPRMC sentence: $GPRMC,time,status,lat,lat_dir,lon,lon_dir,speed,course,date,mag_var,mag_var_dir*checksum
  int commaCount = 0;
  int lastComma = 0;
  String fields[12];
  
  // Split sentence by commas
  for (int i = 0; i < sentence.length(); i++) {
    if (sentence.charAt(i) == ',') {
      if (commaCount < 12) {
        fields[commaCount] = sentence.substring(lastComma, i);
        lastComma = i + 1;
        commaCount++;
      }
    }
  }
  
  // Get the last field (before checksum)
  if (commaCount >= 10) {
    int asteriskPos = fields[10].indexOf('*');
    if (asteriskPos > 0) {
      fields[10] = fields[10].substring(0, asteriskPos);
    }
  }
  
  // Check if we have enough fields and GPS is valid
  if (commaCount >= 10 && fields[2] == "A") { // Status = A (valid)
    gpsTime = fields[1]; // Time (HHMMSS.SSS)
    gpsDate = fields[9]; // Date (DDMMYY)
    gpsTimeValid = true;
    
    // Format time for better readability (HH:MM:SS)
    if (gpsTime.length() >= 6) {
      String formattedTime = gpsTime.substring(0, 2) + ":" + 
                           gpsTime.substring(2, 4) + ":" + 
                           gpsTime.substring(4, 6);
      gpsTime = formattedTime;
    }
    
    // Format date for better readability (DD/MM/YY)
    if (gpsDate.length() >= 6) {
      String formattedDate = gpsDate.substring(0, 2) + "/" + 
                           gpsDate.substring(2, 4) + "/" + 
                           gpsDate.substring(4, 6);
      gpsDate = formattedDate;
    }
    
    debugPrint("GPS Time: " + gpsTime + " Date: " + gpsDate);
  } else {
    gpsTimeValid = false;
  }
}

void testMQTTConnection() {
  if (mqttClient.connected()) {
    debugPrint("Testing MQTT connection...");
    String testPayload = "{\"test\":\"connection\",\"timestamp\":" + String(millis()) + "}";
    
    if (mqttClient.publish(mqttTopic.c_str(), testPayload.c_str())) {
      debugPrint("MQTT test publish successful");
    } else {
      debugPrint("MQTT test publish failed - State: " + String(mqttClient.state()));
    }
  } else {
    debugPrint("MQTT not connected for testing");
  }
}

void loadMQTTTopic() {
  EEPROM.begin(512);
  String storedTopic = "";
  for (int i = 0; i < MQTT_TOPIC_MAX_LEN; i++) {
    char c = EEPROM.read(MQTT_TOPIC_ADDR + i);
    if (c == 0) break;
    storedTopic += c;
  }
  EEPROM.end();
  
  if (storedTopic.length() > 0) {
    mqttTopic = storedTopic;
    debugPrint("Loaded MQTT topic from EEPROM: " + mqttTopic);
  } else {
    debugPrint("Using default MQTT topic: " + mqttTopic);
  }
}

void saveMQTTTopic() {
  EEPROM.begin(512);
  for (int i = 0; i < MQTT_TOPIC_MAX_LEN; i++) {
    if (i < mqttTopic.length()) {
      EEPROM.write(MQTT_TOPIC_ADDR + i, mqttTopic[i]);
    } else {
      EEPROM.write(MQTT_TOPIC_ADDR + i, 0);
    }
  }
  EEPROM.commit();
  EEPROM.end();
  debugPrint("Saved MQTT topic to EEPROM: " + mqttTopic);
}

void handleSetMQTTTopic() {
  if (server.hasArg("mqtt_topic")) {
    String newTopic = server.arg("mqtt_topic");
    if (newTopic.length() > 0 && newTopic.length() <= MQTT_TOPIC_MAX_LEN) {
      mqttTopic = newTopic;
      saveMQTTTopic();
      
      String html = "<!DOCTYPE html><html><head><title>MQTT Topic Updated</title>";
      html += "<meta name='viewport' content='width=device-width, initial-scale=1.0'>";
      html += "<style>";
      html += "body{font-family:Arial,sans-serif;margin:20px;background:#f0f0f0;text-align:center;}";
      html += ".container{max-width:400px;margin:50px auto;background:white;padding:30px;border-radius:10px;box-shadow:0 2px 10px rgba(0,0,0,0.1);}";
      html += "h1{color:#00aa00;}";
      html += "p{margin:15px 0;}";
      html += "a{color:#0066cc;text-decoration:none;}";
      html += "a:hover{text-decoration:underline;}";
      html += "</style></head><body>";
      html += "<div class='container'>";
      html += "<h1>MQTT Topic Updated</h1>";
      html += "<p>MQTT topic has been updated to:</p>";
      html += "<p><strong>" + mqttTopic + "</strong></p>";
      html += "<p>New topic will be used for future MQTT publications.</p>";
      html += "<p><a href='/config'>Back to Configuration</a></p>";
      html += "<p><a href='/'>Back to Dashboard</a></p>";
      html += "</div></body></html>";
      
      server.send(200, "text/html", html);
    } else {
      server.send(400, "text/plain", "Invalid MQTT topic length");
    }
  } else {
    server.send(400, "text/plain", "Missing mqtt_topic parameter");
  }
}

void loadMQTTCredentials() {
  EEPROM.begin(512);
  
  // Load username
  String storedUsername = "";
  for (int i = 0; i < MQTT_CRED_MAX_LEN; i++) {
    char c = EEPROM.read(MQTT_USERNAME_ADDR + i);
    if (c == 0) break;
    storedUsername += c;
  }
  
  // Load password
  String storedPassword = "";
  for (int i = 0; i < MQTT_CRED_MAX_LEN; i++) {
    char c = EEPROM.read(MQTT_PASSWORD_ADDR + i);
    if (c == 0) break;
    storedPassword += c;
  }
  
  EEPROM.end();
  
  if (storedUsername.length() > 0) {
    mqttUsername = storedUsername;
    mqttPassword = storedPassword;
    debugPrint("Loaded MQTT credentials from EEPROM (username: " + mqttUsername + ")");
  } else {
    debugPrint("No MQTT credentials stored, using anonymous connection");
  }
}

void saveMQTTCredentials() {
  EEPROM.begin(512);
  
  // Save username
  for (int i = 0; i < MQTT_CRED_MAX_LEN; i++) {
    if (i < mqttUsername.length()) {
      EEPROM.write(MQTT_USERNAME_ADDR + i, mqttUsername[i]);
    } else {
      EEPROM.write(MQTT_USERNAME_ADDR + i, 0);
    }
  }
  
  // Save password
  for (int i = 0; i < MQTT_CRED_MAX_LEN; i++) {
    if (i < mqttPassword.length()) {
      EEPROM.write(MQTT_PASSWORD_ADDR + i, mqttPassword[i]);
    } else {
      EEPROM.write(MQTT_PASSWORD_ADDR + i, 0);
    }
  }
  
  EEPROM.commit();
  EEPROM.end();
  debugPrint("Saved MQTT credentials to EEPROM");
}

void handleSetMQTTCredentials() {
  if (server.hasArg("mqtt_username") && server.hasArg("mqtt_password")) {
    String newUsername = server.arg("mqtt_username");
    String newPassword = server.arg("mqtt_password");
    
    if (newUsername.length() <= MQTT_CRED_MAX_LEN && newPassword.length() <= MQTT_CRED_MAX_LEN) {
      mqttUsername = newUsername;
      mqttPassword = newPassword;
      saveMQTTCredentials();
      
      String html = "<!DOCTYPE html><html><head><title>MQTT Credentials Updated</title>";
      html += "<meta name='viewport' content='width=device-width, initial-scale=1.0'>";
      html += "<style>";
      html += "body{font-family:Arial,sans-serif;margin:20px;background:#f0f0f0;text-align:center;}";
      html += ".container{max-width:400px;margin:50px auto;background:white;padding:30px;border-radius:10px;box-shadow:0 2px 10px rgba(0,0,0,0.1);}";
      html += "h1{color:#00aa00;}";
      html += "p{margin:15px 0;}";
      html += "a{color:#0066cc;text-decoration:none;}";
      html += "a:hover{text-decoration:underline;}";
      html += "</style></head><body>";
      html += "<div class='container'>";
      html += "<h1>MQTT Credentials Updated</h1>";
      html += "<p>MQTT credentials have been saved.</p>";
      html += "<p><strong>Username:</strong> " + (mqttUsername.length() > 0 ? mqttUsername : "(none)") + "</p>";
      html += "<p>The ESP32 will reconnect to MQTT with the new credentials.</p>";
      html += "<p><a href='/config'>Back to Configuration</a></p>";
      html += "<p><a href='/'>Back to Dashboard</a></p>";
      html += "</div></body></html>";
      
      server.send(200, "text/html", html);
      
      // Reconnect MQTT with new credentials
      mqttClient.disconnect();
      delay(1000);
      reconnectMQTT();
    } else {
      server.send(400, "text/plain", "Invalid credentials length");
    }
  } else {
    server.send(400, "text/plain", "Missing parameters");
  }
}

void debugPrint(const String& message) {
  if (DEBUG_ENABLED) {
    Serial.println("[" + String(millis()) + "] " + message);
  }
}
